package com.system.oauth.message.websocket;

import com.alibaba.fastjson.JSONObject;
import com.system.framework.core.constant.WebsocketConst;
import com.system.framework.core.utils.oConvertUtils;
import com.system.oauth.message.handle.enums.SendMsgLinkEnum;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.websocket.*;
import javax.websocket.server.PathParam;
import javax.websocket.server.ServerEndpoint;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @Author Lyrical
 * @Date 2022/04/27
 * @Description: 此注解相当于设置访问URL
 */
@Component
@Slf4j
@ServerEndpoint(value = "/websocket/{userId}/{client}", subprotocols = {"protocol"}) //此注解相当于设置访问URL
public class Socket {
    private static int onlineCount = 0;
    //map<key,this.class>
    private static ConcurrentHashMap<String, Socket> webSocketSet = new ConcurrentHashMap<>();
    //map<id:list<key>>
    private static ConcurrentHashMap<String, List<String>> map = new ConcurrentHashMap<>();

    //与某个客户端的连接会话，需要通过它来给客户端发送数据
    private Session session;

    private String id = "";
    //   用户与客户端标识 id:{client}
    private String key = "";


    /**
     * 连接建立成功调用的方法
     */
    @OnOpen
    public void onOpen(@PathParam(value = "userId") String id, @PathParam("client") String client, Session session) {
        System.out.println(id);
        this.session = session;
        this.id = id;//接收到发送消息的人员编号
        this.key = id + ":" + SendMsgLinkEnum.getClient(client);
        //非SendMsgLinkEnum client 拒接
        if (oConvertUtils.isEmpty(SendMsgLinkEnum.getClient(client))) {
            log.info("非正式客户端连接！");
            try {
                this.sendMessage("非正式客户端连接");
                this.session.close();
            } catch (IOException e) {
                e.printStackTrace();
            } finally {
                return;
            }

        } else {
//            String key = id + ":" + SendMsgLinkEnum.getClient(client);
            List<String> ids = new ArrayList<>();
            if (!map.isEmpty() || map.containsKey(id)) {
                ids = map.get(id);
            }

            //避免出现重复的 key
            if (!ids.contains(key)) {
                ids.add(key);
            }

//            ids.add(key);
            map.put(id, ids);
            webSocketSet.put(key, this);     //加入set中
        }
        addOnlineCount();           //在线数加1
        log.info("用户" + id + "连接成功！当前在线人数为" + getOnlineCount());
        try {
            sendMessage("连接成功");

//            System.out.println(webSocketSet.size());
//            System.out.println(map.size());
//            System.out.println(webSocketSet);
//            System.out.println(map);
        } catch (IOException e) {
            log.error("websocket IO异常");
        }

    }

    /**
     * 连接关闭调用的方法
     */
    @OnClose
    public void onClose() {
        if (!map.isEmpty() || map.containsKey(id)) {
            //删除map中值
            List<String> keys = map.get(this.id);
            for (int i = 0; i < keys.size(); i++) {
                if (this.key.equals(keys.get(i))) {
                    keys.remove(i);
                    i--;
//                    if (keys.size() == 0) {
//                        map.remove(this.id);
//                    }
                }

            }

            webSocketSet.remove(this.key);  //从set中删除
            subOnlineCount();           //在线数减1
        }

        log.info("【websocket消息】连接断开，当前在线人数为" + getOnlineCount());
    }

    /**
     * 收到客户端消息后调用的方法
     *
     * @param message 客户端发送过来的消息
     */
    @OnMessage
    public void onMessage(String message) {
        //todo 现在有个定时任务刷，应该去掉
        log.info("【websocket消息】收到客户端消息:" + message);
        JSONObject obj = new JSONObject();
        obj.put(WebsocketConst.MSG_CMD, WebsocketConst.CMD_CHECK);//业务类型
        obj.put(WebsocketConst.MSG_TXT, "心跳响应");//消息内容
        session.getAsyncRemote().sendText(obj.toJSONString());
    }

    /**
     * @param session
     * @param error
     */
    @OnError
    public void onError(Session session, Throwable error) {
        log.error("【websocket消息】" + this.id + " 发生错误");
        try {
            session.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
        error.printStackTrace();

    }

    /**
     * 服务端主动推送消息
     *
     * @param message
     * @throws IOException
     */
    public void sendMessage(String message) throws IOException {
        if (session.isOpen())
            this.session.getBasicRemote().sendText(message);
    }

    /**
     * 发送信息给指定ID用户，如果用户不在线则返回不在线信息给自己
     *
     * @param message
     * @param userId
     * @throws IOException
     */
    public static void sendOneMessage(String userId, String message) {
        //获取map中对应的链接
        if (map.get(userId) != null && oConvertUtils.isNotEmpty(map.get(userId))) {
            //遍历给连接人发送信息
            for (String s : map.get(userId)) {
                log.info("【websocket消息】 单点消息:" + message);
                try {
                    webSocketSet.get(s).sendMessage(message);
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        } else {
            //如果用户不在线则返回不在线信息给自己
            log.error("当前用户不在线：" + userId);
        }


    }

    /**
     * 发送信息给指定ID用户集合，如果用户不在线则返回不在线信息给自己
     *
     * @param message
     * @param userIds
     * @throws IOException
     */
    public static void sendMoreMessage(String message, String[] userIds) throws IOException {
        //获取人员集合
        for (String userId : userIds) {
            //获取map中对应的链接
            if (map.get(userId) != null && oConvertUtils.isNotEmpty(map.get(userId))) {
                //遍历给连接人发送信息
                for (String s : map.get(userId)) {
                    webSocketSet.get(s).sendMessage(message);
                }
            } else {
                //如果用户不在线则返回不在线信息给自己
                log.error("【websocket消息】 当前用户不在线：" + userId);
            }
        }

    }

    /**
     * 发送信息给所有人
     *
     * @param
     * @throws IOException
     */
    public void sendAllMessage(String message) {
        for (String key : webSocketSet.keySet()) {
            try {
                log.info("【websocket消息】 单点消息:" + message);
                webSocketSet.get(key).sendMessage(message);
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }


    public static synchronized int getOnlineCount() {
        return onlineCount;
    }

    public static synchronized void addOnlineCount() {
        Socket.onlineCount++;
    }

    public static synchronized void subOnlineCount() {
        Socket.onlineCount--;
    }
}
